/**
 * Class ........: RemoteDatabaseServer
 * Description ..: This is the remote server process for integration with the remote database.
 *                 The methods of this class will be called from the AS/400 using RMI.
 *
 * @author Liam McMahon
 * @version 1.0
 */


import java.sql.*;
import java.rmi.*;
import java.rmi.server.*;
import java.rmi.registry.*;
import java.io.*;


public class RemoteDatabaseServer extends UnicastRemoteObject implements RemoteDatabaseServerInf {

      Connection conn;
      Statement stmt;
      ResultSet rs;


      /**
       * Class constructor.  Sets the security manager and gets a connection to the database.
       */

      public RemoteDatabaseServer() throws RemoteException {
            super();

            /**
             * Set the security manager to the RMI security manager, to prevent the client
             * process from using the server in a malicious way.
             */

            try {
                  if (System.getSecurityManager() == null) {
                        System.setSecurityManager(new RMISecurityManager());
                        System.out.println("RMI security manager loaded!");
                  }
            }
            catch(Exception e) {
                  System.out.println("Security violation : " + e);
            }

            /**
             * Open a connection to the database.
             */

            getDatabaseConnection();
      }


      /**
       * Get a connection to the database, if a connection was not already created.
       */

      public void getDatabaseConnection() {

            if (conn == null) {
                  try {
                        DriverManager.registerDriver(new sun.jdbc.odbc.JdbcOdbcDriver());
                        conn = DriverManager.getConnection("jdbc:odbc:RemoteDatabase");
                        System.out.println("Connection to the database is open...");
                  }
                  catch(Exception e) {
                        System.out.println("Error connecting to the database : " + e);
                        conn = null;
                  }
            }
      }


      /**
       * This method is called from the AS/400 using RMI.  It will accept an Invoice Number and
       * a Purchase Order Number as arguments.  It will then attempt to retrieve that invoice
       * from the database.  If the invoice is found then the method will create an object of
       * type Invoice, to represent the actual invoice.  The Invoice object is then returned to
       * the caller.  If the invoice is not found in the database then a null reference is
       * returned to the caller.
       */

      public synchronized Invoice getInvoice(int invoice) throws RemoteException {

            Invoice inv = null;

            try {

                  /**
                   * Create an SQL statement to get the invoice details, and execute it.
                   */

                  stmt = conn.createStatement();

		  String sqlString = "Select INV_VAL, INV_CUR, INV_STS from INVOICES " +
		  	                "where INV_REF = " + invoice;

		  rs = stmt.executeQuery(sqlString);

                  /**
                   * If the record was found, extract the invoice details from the result set
                   * and create an invoice object to return to the calling process.
                   */

                  if (rs.next()) {
                        inv = new Invoice(invoice,
                                          rs.getDouble("INV_VAL"),
                                          rs.getString("INV_CUR"),
					  rs.getString("INV_STS"));
                  }


                  /**
                   * Tidy up.  Close the result set and the statement.  The connection to the
                   * database won't be closed; we want that to stay open so that it is available
                   * for the next caller.
                   */

                  rs.close();
                  stmt.close();
            }
            catch(Exception e) {
                  System.out.println("Error occurred when getting the invoice: " + e);
            }

            return inv;
      }


      /**
       * The main() method will create an instance of this class and then bind it to the RMI
       * registry.
       */

      public static void main(String[] args) throws Exception {

            String ipAddress = "//192.168.28.20/";


            /**
             * Start the RMI registry.
             */

            try {
                  LocateRegistry.createRegistry(Registry.REGISTRY_PORT);
            }
            catch (RemoteException re) {
                  System.out.println("Error occurred when starting the RMI registry: " + re);
                  return;
            }

            System.out.println("\n\nStarting the database server process.");


            /**
             * Create an instance of the server class, and bind it to the registry.
             */

            RemoteDatabaseServer rmtServerProcess = new RemoteDatabaseServer();
            if (rmtServerProcess.conn == null) {
                  System.out.println("\n\nThe database server process has NOT been successfully started.");

		  System.exit(0);
            }

            else {
                  System.out.println("Binding process " + ipAddress + "DATABASE-SERVER to the RMI registry");

		  try {
                  	Naming.bind(ipAddress + "DATABASE-SERVER", rmtServerProcess);
                        System.out.println("Database server process is now bound to rmi registry!");
                        System.out.println("\n\nThe database server process has been successfully started.");
		  }
  		  catch(Exception e) {
                        System.out.println("Database server process could not be bound to the rmi registry!");
			System.out.println("Error : " + e);

			System.exit(0);
		  }
            }
      }
}