/**
 * Class ........: RemoteDatabaseClient
 * Description ..: This is the AS/400 side Java stored procedure that will be called from
 *                 the RPG IV SQL program.  The program will get an invoice object from the
 *                 remote database using RMI.
 *
 * @author Liam McMahon
 * @version 1.0
 */


import java.rmi.*;
import java.rmi.server.*;

public class RemoteDatabaseClient {


      /**
       * Get the details of an invoice from the remote database, by calling a remote method
       * on the database server using RMI.  An object of type Invoice will be returned from
       * the called remote method. This method uses the Java parameter style for calling the
       * stored procedure, so the method must be static, and the OUT and INOUT parameters are
       * defined as arrays.  To return a value to the caller, you put the value in the first
       * element of the array.
       */

      public static void getRemoteInvoice(int[] rmtInvoice, double rmtInvoiceVal[],
                                          String[] rmtInvoiceCurr, String[] rmtInvoiceSts,
                                          String[] rmtErrorCode, String[] rmtErrorMsg) {

            /**
             * If the method was called with Invoice Number of zero, assume that the call is just
             * a dummy to load the JVM.  Return to the calling program immediately.
             */

            if (rmtInvoice[0] == 0)
                  return;


            /**
             * Set the security manager to the RMI security manager.
             */

            if (!RemoteDatabaseClient.setSecurityManager()) {
                  rmtErrorCode[0] = "00001";
                  rmtErrorMsg[0] = "A Java security violation occurred.";
                  return;
            }

            /**
             * Get a reference to the RemoteDatabaseServer class that should be running on the
             * remote server.
             */

            RemoteDatabaseServerInf rmtServerProcess = RemoteDatabaseClient.getRemoteReference();
            if (rmtServerProcess == null) {
                  rmtErrorCode[0] = "00002";
                  rmtErrorMsg[0] = "Could not contact the remote server process. Possible network problem";
                  return;
            }


            /**
             * Initialise the Invoice object reference.
             */

            Invoice inv = null;


            /**
             * Call the getInvoice method on the remote server to get the details of the invoice
             * that we need.  An object of type Invoice will be returned to this program.
             */

            try {
                  inv = rmtServerProcess.getInvoice(rmtInvoice[0]);
            }
            catch(Exception e) {
                  rmtErrorCode[0] = "00003";
                  rmtErrorMsg[0] = "Could not get the Invoice from the database. Possible network problem";
                  return;
            }


            /**
             * If a null Invoice object was returned from the remote method invocation then set
             * the error code and error message to return to the caller.  Otherwise return the
             * Invoice object's values to the caller.
             */

             if (inv == null) {
                  rmtInvoiceVal[0]  = 0;
                  rmtInvoiceCurr[0] = " ";
                  rmtInvoiceSts[0] = " ";
                  rmtErrorCode[0]   = "00004";
                  rmtErrorMsg[0]    = "The Invoice was not found in the remote database";
             }
             else {
                  rmtInvoiceVal[0]  = inv.invoiceVal;
                  rmtInvoiceCurr[0] = inv.invoiceCurr;
                  rmtInvoiceSts[0]  = inv.invoiceSts;
                  rmtErrorCode[0]   = " ";
                  rmtErrorMsg[0]    = " ";
             }
      }


      /**
        * Set the security manager to the RMI security manager.
        */

      public static boolean setSecurityManager() {

            try {
                  if (System.getSecurityManager() == null) {
                        System.setSecurityManager(new RMISecurityManager());
                  }
            }
            catch(Exception e) {
                  return false;
            }

            return true;
      }


      /**
        * Get a reference to the RemoteDatabaseServer class that should be running on the
        * remote server.
        */

      public static RemoteDatabaseServerInf getRemoteReference() {

	    String ipAddress = "//192.168.28.20/";
            RemoteDatabaseServerInf rmtServerProcess = null;

            try {
                 rmtServerProcess = (RemoteDatabaseServerInf)Naming.lookup(ipAddress + "DATABASE-SERVER");
                 return rmtServerProcess;
            }
            catch(Exception e) {
                  return null;
            }
      }
}